#include <array>
#include <iostream>

#include "phylogenetics.hpp"
#include "unit_test.hpp"
#include "utils.hpp"

using namespace std;

void test_part1();
void test_part2();
void test_part3();
void testPhylogeneticTree();
void test_merge_clusters(Tree& tree, DistanceMatrix& distances,
                         ClusterIdPair pair, AlgoType algoType);

string data_dir = "../data/";

int main(int argc, const char* argv[]) {
    cout << "========= TESTING PART 1 ================" << endl;
    //test_part1();
    cout << "========= TESTING PART 2 ================" << endl;
    //test_part2();
    cout << "========= TESTING PART 3 ================" << endl;
    //test_part3();
    cout << "========= UNIT TESTS ================" << endl;
    //run_unit_tests(1);
    //run_unit_tests(2);
    //run_unit_tests(3);

    return 0;
}

void testPhylogeneticTree() {
    cout << " ======= Testing build and print phylogenetic trees ======= "
         << endl;
    vector<string> elements =
        readSequencesFromFile(data_dir + "dna_sequences.txt");
    cout << "Input data:" << endl;
    Tree tree = initTree(elements);
    cout << "Tree:" << endl;
    cout << "------" << std::endl;
    cout << "Verbose printing:" << endl;
    cout << toString(tree, true);
    cout << "Non verbose printing:" << endl;
    cout << toString(tree);
    DistanceMatrix distances = initDistanceMatrix(tree);
    cout << "\nDistance matrix:" << endl;
    cout << "-----------------" << endl;
    cout << toString(distances);

    cout << "----- Build Phylogenetic Tree with WPGMA ----" << endl;
    /* tested with the non final API (buildPhylogeneticTree from Tree
       and not from a vector of taxa)
    */
    buildPhylogeneticTree(tree, distances);
    std::cout << "\nAfter calling buildPhylogeneticTree:" << endl;
    cout << "Tree:" << endl;
    cout << "------" << endl;
    cout << toString(tree);
    cout << "\nDistance matrix:" << endl;
    cout << "-----------------" << endl;
    cout << toString(distances);
    std::cout << "\nCalling phylogeneticTreeToString:" << endl;
    cout << endl;
    cout << "Verbose printing:" << endl;
    cout << phylogeneticTreeToString(tree.at(0), true);
    cout << "Non verbose printing:" << endl;
    cout << phylogeneticTreeToString(tree.at(0));
    deleteTree(tree);

    cout << endl;
    cout << "----- Build Phylogenetic Tree with UPGMA ----" << endl;
    Tree tree2 = initTree(elements);
    DistanceMatrix distances2 = initDistanceMatrix(tree2);
    buildPhylogeneticTree(tree2, distances2, UPGMA);
    std::cout << "\nAfter calling buildPhylogeneticTree:" << endl;
    cout << endl;
    cout << "Tree:" << endl;
    cout << "------" << endl;
    cout << toString(tree2);
    std::cout << "\nCalling phylogeneticTreeToString:" << endl;
    cout << endl;
    cout << "Verbose printing:" << endl;
    cout << phylogeneticTreeToString(tree2.at(0), true);
    cout << "Non verbose printing:" << endl;
    cout << phylogeneticTreeToString(tree2.at(0));
    deleteTree(tree2);

    // cout << "Testing buildPhylogeneticTree(const std::vector<Taxon>& ...)"
    //      << endl;
    // // correct/final  API (the other one is introduced to facilitate tests):
    // Tree treeFromTaxa(buildPhylogeneticTree(elements, UPGMA));
    // cout << toString(treeFromTaxa);
    // cout << phylogeneticTreeToString(treeFromTaxa[0], true);
    // deleteTree(treeFromTaxa);

    cout << "----- Wikipedia Example with UPGMA ----" << endl;
    vector<Taxon> items = {"a", "b", "c", "d", "e"};
    Tree tree3 = initTree(items);
    DistanceMatrix dist3 = {{0, 17, 21, 31, 23},
                            {17, 0, 30, 34, 21},
                            {21, 30, 0, 28, 39},
                            {31, 34, 28, 0, 43},
                            {23, 21, 39, 43, 0}};
    buildPhylogeneticTree(tree3, dist3, UPGMA);
    cout << toString(tree3);
    cout << phylogeneticTreeToString(tree3[0], true);
    deleteTree(tree3);

    cout << "----- Wikipedia Example with WPGMA ----" << endl;
    Tree tree4 = initTree(items);
    DistanceMatrix dist4 = {{0, 17, 21, 31, 23},
                            {17, 0, 30, 34, 21},
                            {21, 30, 0, 28, 39},
                            {31, 34, 28, 0, 43},
                            {23, 21, 39, 43, 0}};
    buildPhylogeneticTree(tree4, dist4, WPGMA);
    cout << toString(tree4, true);
    cout << phylogeneticTreeToString(tree4[0], true);
    deleteTree(tree4);
    cout << " ======= end testing build and print phylogenetic trees ======= "
         << endl;
}

void test_part1() {
    cout << " ======= Testing toString for DistanceMatrix ======= " << endl;
    cout << " ======= printing in non verbose mode by default ====" << endl;
    DistanceMatrix dist = {{0, 17, 21, 31, 23},
                           {17, 0, 30, 34, 21},
                           {21, 30, 0, 28, 39},
                           {31, 34, 28, 0, 43},
                           {23, 21, 39, 43, 0}};
    cout << toString(dist);
    cout << " ======= printing in verbose mode ====" << endl;
    cout << toString(dist, true);
    cout << " ======= End testing toString for DistanceMatrix ======= " << endl;
    cout << " ======= Testing toString for ClusterIdPair ======= " << endl;
    ClusterIdPair id_pair({2,3});
    cout << toString(id_pair);
    cout << " ======= End testing toString for ClusterIdPair ======= " << endl;
    cout << " ======= Testing clusterToString ======= " << endl;
    Cluster t1 = {"ACGTAACCTTGGG", 0, 0.0, nullptr, nullptr, 1};
    Cluster t2 = {"AGGGTCTATATGT", 1, 0.0, nullptr, nullptr, 1};
    Cluster t3 = {"GTAGTAGTAGTAG", 2, 0.0, nullptr, nullptr, 1};
    Cluster t1t3 = {"", -1, 2.0, &t1, &t3, 2};
    Cluster t1t3t2 = {"", -1, 3.0, &t1t3, &t2, 3};
    cout << " ======= printing in non verbose mode by default ====" << endl;
    cout << clusterToString(&t1t3t2);
    cout << endl;
    cout << " ======= printing in verbose mode  ====" << endl;
    cout << clusterToString(&t1t3t2, true);
    cout << endl;
    cout << " ======= End testing clusterToString ======= " << endl;

    cout << " ======= Testing toString for Tree ======= " << endl;
    Tree tree1({&t1, &t2});
    cout << "printing tree1:" << endl;
    cout << "---------------" << endl;
    cout << " ======= printing in verbose mode  ====" << endl;
    cout << toString(tree1, true);
    cout << " ======= printing in non verbose mode  ====" << endl;
    cout << toString(tree1);

    Tree tree2 = {&t1t3t2};
    cout << "printing tree2:" << endl;
    cout << "---------------" << endl;
    cout << " ======= printing in verbose mode  ====" << endl;
    cout << toString(tree2, true);
    cout << " ======= printing in non verbose mode  ====" << endl;
    cout << toString(tree2);
    cout << " ======= End testing toString for Tree ======= " << endl;
    cout << endl;
}

void test_part2() {
    cout << " ======= Testing calculateDistance ======= " << endl;
    Taxon taxon1("CGTAACCTTGGG");
    Taxon taxon2("CGTGAGCTTA");
    cout << "Distance between " << taxon1 << " and " << taxon2 << ": ";
    cout << calculateDistance(taxon1, taxon2) << endl;
    cout << " ======= end Testing calculateDistance ======= " << endl;

    cout << " ======= Testing initTree ======= " << endl;
    vector<Taxon> items = {"a", "b", "c", "d", "e"};
    cout
        << "The initial vector of taxa is: {\"a\", \"b\", \"c\", \"d\", \"e\"} "
        << endl;
    Tree tree1 = initTree(items);
    std::cout << "Tree constructed by initTree, printed in non verbose mode:"
              << endl;
    cout << toString(tree1);
    std::cout << "Same tree printed in verbose mode:" << endl;
    cout << toString(tree1, true);
    cout << " ======= end testing initTree ======= " << endl;

    cout << " ======= Testing initDistanceMatrix ======= " << endl;
    Cluster t1 = {"ACGTAACCTTGGG", 0, 0.0, nullptr, nullptr, 1};
    Cluster t2 = {"ACGGTCTATTGGA", 1, 0.0, nullptr, nullptr, 1};
    Cluster t3 = {"GTAGTAGTAGTAG", 2, 0.0, nullptr, nullptr, 1};
    cout << "Input data: a tree constructed using the taxa ";
    cout << t1.taxon << ", " << t2.taxon << " and  " << t3.taxon << endl;
    Tree tree2 = {&t1, &t2, &t3};
    std::cout << "Distance matrix constructed by initDistanceMatrix:" << endl;
    cout << toString(initDistanceMatrix(tree2));
    cout << " ======= end testing initDistanceMatrix ======= " << endl;

    cout << " ======= Testing eraseColumn and eraseRow ======= " << endl;
    cout << "Initial distance matrix:" << endl;

    DistanceMatrix distances = {{0, 17, 21, 31, 23},
                                {17, 0, 30, 34, 21},
                                {21, 30, 0, 28, 39},
                                {31, 34, 28, 0, 43},
                                {23, 21, 39, 43, 0}};
    cout << toString(distances);

    cout << "The previous matrix after erasing column 3:" << endl;
    eraseColumn(distances, 3);
    cout << toString(distances);

    cout << "The previous matrix after erasing row 3:" << endl;
    eraseRow(distances, 3);
    cout << toString(distances);
    cout << " ======= end testing eraseColumn and eraseRow ======= " << endl;
    cout << endl;
}

void test_part3() {
    cout << " ======= Testing minimumDistance ======= " << endl;
    cout << "Input data: the distance matrix" << endl;

    DistanceMatrix distance = {{0, 1, 2}, {1, 0, 1}, {2, 1, 0}};

    cout << "Input data: the distance matrix  {0, 1, 2}, {1, 0, 1}, {2, 1, 0}"
         << endl;
    ClusterIdPair pair = minimumDistance(distance);
    cout << "The pair with the minimum distance is:" << endl;
    cout << pair[0] << "-" << pair[1] << endl;
    cout << " ======= End testing minimumDistance ======= " << endl;

    cout << " ======= Testing mergeCluster (WPGMA) ======= " << endl;
    Cluster t1 = {"ACGTAACCTTGGG", 0, 0.0, nullptr, nullptr, 1};
    Cluster t2 = {"ACGGTCTATTGGA", 1, 0.0, nullptr, nullptr, 1};
    Cluster t3 = {"GTAGTAGTAGTAG", 2, 0.0, nullptr, nullptr, 1};
    Tree tree = {&t1, &t2, &t3};
    DistanceMatrix distances({{0, 6, 11}, {6, 0, 11}, {11, 11, 0}});
    pair = {0, 1};
    test_merge_clusters(tree, distances, pair, WPGMA);
    cout << " ======= End testing mergeCluster (WPGMA) ======= " << endl;

    cout << " ======= Testing mergeCluster (UPGMA) ======= " << endl;
    cout << "Input data: a tree with three leaf nodes \"ACGTAACCTTGGG\", "
            "\"ACGGTCTATTGGA\"";
    cout << " and \"GTAGTAGTAGTAG\"  " << endl;
    tree = {&t1, &t2, &t3};
    distances = {{0, 6, 11}, {6, 0, 11}, {11, 11, 0}};
    pair = {0, 1};
    test_merge_clusters(tree, distances, pair, UPGMA);
    cout << " ======= End testing mergeCluster (UPGMA) ======= " << endl;

    // Phylogenetic tree (building+printing)
    testPhylogeneticTree();
}

void test_merge_clusters(Tree& tree, DistanceMatrix& distances,
                         ClusterIdPair pair, AlgoType algoType) {
    cout << "\nThe pair to merge:" << endl;
    cout << toString(pair);
    cout << "\nBefore merging:" << endl;
    cout << "-----------------" << endl;
    cout << "Tree:" << endl;
    cout << "------" << endl;
    cout << toString(tree, true);
    cout << "\nDistance matrix:" << endl;
    cout << "-----------------" << endl;
    cout << toString(distances);

    mergeClusters(pair, tree, distances, algoType);

    cout << "\nAfter merging clusters:" << endl;
    cout << "------------------------------" << endl;

    cout << "Tree:" << endl;
    cout << "------" << endl;
    cout << toString(tree, true);
    cout << "\nDistance matrix:" << endl;
    cout << "-----------------" << endl;
    cout << toString(distances);
    cout << endl;
}
